<?php

declare(strict_types=1);

namespace Bridge\Database\Factories;

use Bridge\Domains\Currency\Currency;
use Bridge\Domains\Invoice\Enums\InvoiceType;
use Bridge\Domains\Invoice\Models\Invoice;
use Bridge\Domains\Invoice\Models\InvoiceItem;
use Bridge\Domains\Item\Enums\ItemType;
use Bridge\Domains\Item\Models\Item;
use Bridge\Domains\Item\Models\ItemPrice;
use Closure;
use Illuminate\Database\Eloquent\Factories\Factory;

/**
 * @extends \Illuminate\Database\Eloquent\Factories\Factory<InvoiceItem>
 */
class InvoiceItemFactory extends Factory
{
    /**
     * @return array<string, mixed>
     */
    public function definition(): array
    {
        return [
            'order' => 0,
            'invoice_id' => Invoice::factory(),
            'item_id' => Item::factory()->state([
                'type' => fake()->randomElement(ItemType::options()),
            ]),

            'price' => $this->generateItemPrice(),
            'quantity' => $this->generateQuantity(),
            'description' => fake()->optional(0.2)->sentence(),

            'currency_id' => Currency::factory(),
            'currency_rate' => $this->generateCurrencyRate(),

            'discount' => 0,
            'discount_type' => null,
            'discount_amount' => null,
        ];
    }

    private function generateItemPrice(): Closure
    {
        return function (array $attributes) {
            // 'invoice_type' is not a column in the table, but it's used during testing
            $invoiceType = $attributes['invoice_type'] ?? Invoice::query()
                ->where('id', $attributes['invoice_id'])
                ->soleValue('type');

            $itemPrice = ItemPrice::query()
                ->where('item_id', $attributes['item_id'])
                ->sole();

            $price = $invoiceType === InvoiceType::Sale
                ? $itemPrice->sell_for
                : $itemPrice->buy_for;

            $price = !empty($price) ? $price : fake()->randomFloat(2, 100, 1000);

            return fake()->boolean(90)
                ? $price
                : $price + fake()->numberBetween(10, 20);
        };
    }

    public function generateCurrencyRate(): Closure
    {
        return function (array $attributes) {
            $rate = Currency::query()
                ->where('id', $attributes['currency_id'])
                ->soleValue('rate');

            return fake()->boolean(90)
                ? $rate
                : $rate + fake()->numberBetween($rate, $rate * 1.5);
        };
    }

    private function generateQuantity(): int
    {
        return fake()->boolean(80)
            ? 1 : fake()->numberBetween(1, 4);
    }
}
