<?php

declare(strict_types=1);

namespace Bridge\Database\Factories;

use Bridge\Domains\Currency\Currency;
use Illuminate\Database\Eloquent\Factories\Factory;
use Illuminate\Support\Str;
use Squire\Models\Currency as SquireCurrency;

/**
 * @extends \Illuminate\Database\Eloquent\Factories\Factory<Currency>
 */
class CurrencyFactory extends Factory
{
    protected $model = Currency::class;

    /**
     * Define the model's default state.
     *
     * @return array<string, mixed>
     */
    public function definition(): array
    {
        $currency = $this->randomUniqueCurrency();

        return [
            'symbol' => $currency->symbol,
            'name' => Str::limit($currency->name, 15),
            // @phpstan-ignore-next-line
            'code' => strtoupper($currency->code_alphabetic),
            'rate' => (string) fake()->randomFloat(4, min: 1, max: 150),
        ];
    }

    public function usd(): self
    {
        return $this->state([
            'name' => 'US Dollar',
            'code' => 'USD',
            'symbol' => '$',
            'rate' => 1,
        ]);
    }

    protected function randomUniqueCurrency(): SquireCurrency
    {
        do {
            $currencyCode = strtolower(fake()->unique()->currencyCode());

            $validCurrency = SquireCurrency::where('code_alphabetic', $currencyCode)->exists();
            $alreadyExists = Currency::query()->where('code', strtoupper($currencyCode))->exists();
        } while (!$validCurrency || $alreadyExists);

        return SquireCurrency::findOrFail($currencyCode);
    }
}
